<?php
/**
 * Validation Utility Class
 * Provides input validation methods
 */

class Validator {
    /**
     * Validate required fields
     * @param array $data Input data
     * @param array $requiredFields List of required field names
     * @return array Array of validation errors (empty if valid)
     */
    public static function validateRequired($data, $requiredFields) {
        $errors = [];
        
        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || trim($data[$field]) === '') {
                $errors[$field] = ["The {$field} field is required"];
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate email format
     * @param string $email Email address
     * @return bool True if valid, false otherwise
     */
    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }
    
    /**
     * Validate mobile number (10 digits)
     * @param string $mobile Mobile number
     * @return bool True if valid, false otherwise
     */
    public static function validateMobile($mobile) {
        return preg_match('/^[0-9]{10}$/', $mobile);
    }
    
    /**
     * Validate age (1-150)
     * @param int $age Age value
     * @return bool True if valid, false otherwise
     */
    public static function validateAge($age) {
        return is_numeric($age) && $age >= 1 && $age <= 150;
    }
    
    /**
     * Validate date format (YYYY-MM-DD or YYYY-MM-DD HH:mm:ss)
     * @param string $date Date string
     * @return bool True if valid, false otherwise
     */
    public static function validateDate($date) {
        $dateTime = DateTime::createFromFormat('Y-m-d H:i:s', $date);
        if ($dateTime) {
            return true;
        }
        
        $dateOnly = DateTime::createFromFormat('Y-m-d', $date);
        return $dateOnly !== false;
    }
    
    /**
     * Sanitize string input
     * @param string $input Input string
     * @return string Sanitized string
     */
    public static function sanitizeString($input) {
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * Validate enum value
     * @param string $value Value to check
     * @param array $allowedValues Allowed values
     * @return bool True if valid, false otherwise
     */
    public static function validateEnum($value, $allowedValues) {
        return in_array($value, $allowedValues);
    }
}

