<?php
/**
 * API Routes Definition
 * Defines all API endpoints and their corresponding controllers
 */

class Router {
    private $routes = [];
    
    /**
     * Add route to routes array
     * @param string $method HTTP method (GET, POST, PUT, DELETE)
     * @param string $path Route path
     * @param string $controller Controller class name
     * @param string $action Controller method name
     * @param bool $requireAuth Whether authentication is required
     */
    private function addRoute($method, $path, $controller, $action, $requireAuth = true) {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'controller' => $controller,
            'action' => $action,
            'auth' => $requireAuth
        ];
    }
    
    /**
     * Register all API routes
     */
    public function registerRoutes() {
        // Authentication routes (no auth required)
        $this->addRoute('POST', '/auth/login', 'AuthController', 'login', false);
        $this->addRoute('POST', '/auth/logout', 'AuthController', 'logout', true);
        
        // Patient routes (order matters - exact routes should be registered first)
        $this->addRoute('GET', '/patients', 'PatientController', 'index', true);
        $this->addRoute('GET', '/patients/search', 'PatientController', 'search', true);
        $this->addRoute('GET', '/patients/last-registration-number', 'PatientController', 'getLastRegistrationNumber', true);
        $this->addRoute('GET', '/patients/latest-registration-number', 'PatientController', 'getLastRegistrationNumber', true);
        $this->addRoute('GET', '/patients/{id}', 'PatientController', 'show', true);
        $this->addRoute('POST', '/patients', 'PatientController', 'create', true);
        $this->addRoute('PUT', '/patients/{id}', 'PatientController', 'update', true);
        $this->addRoute('DELETE', '/patients/{id}', 'PatientController', 'delete', true);
        
        // Doctor routes
        $this->addRoute('GET', '/doctors', 'DoctorController', 'index', true);
        $this->addRoute('GET', '/doctors/{id}', 'DoctorController', 'show', true);
        $this->addRoute('POST', '/doctors', 'DoctorController', 'create', true);
        $this->addRoute('PUT', '/doctors/{id}', 'DoctorController', 'update', true);
        $this->addRoute('DELETE', '/doctors/{id}', 'DoctorController', 'delete', true);
        
        // Staff routes
        $this->addRoute('GET', '/staff', 'StaffController', 'index', true);
        $this->addRoute('GET', '/staff/{id}', 'StaffController', 'show', true);
        $this->addRoute('POST', '/staff', 'StaffController', 'create', true);
        $this->addRoute('PUT', '/staff/{id}', 'StaffController', 'update', true);
        $this->addRoute('DELETE', '/staff/{id}', 'StaffController', 'delete', true);
        
        // Case Sheet routes
        $this->addRoute('GET', '/case-sheets/patient/{patient_id}', 'CaseSheetController', 'getByPatient', true);
        $this->addRoute('GET', '/case-sheets/{id}', 'CaseSheetController', 'show', true);
        $this->addRoute('POST', '/case-sheets', 'CaseSheetController', 'create', true);
        $this->addRoute('PUT', '/case-sheets/{id}', 'CaseSheetController', 'update', true);
        $this->addRoute('DELETE', '/case-sheets/{id}', 'CaseSheetController', 'delete', true);
        
        // Category routes
        $this->addRoute('GET', '/categories', 'CategoryController', 'index', true);
        $this->addRoute('GET', '/categories/parent/{parent_id}', 'CategoryController', 'getByParent', true);
        $this->addRoute('GET', '/categories/{id}', 'CategoryController', 'show', true);
        $this->addRoute('POST', '/categories', 'CategoryController', 'create', true);
        $this->addRoute('PUT', '/categories/{id}', 'CategoryController', 'update', true);
        $this->addRoute('DELETE', '/categories/{id}', 'CategoryController', 'delete', true);
    }
    
    /**
     * Match request to route and execute
     */
    public function dispatch() {
        $requestMethod = $_SERVER['REQUEST_METHOD'];
        $requestUri = $_SERVER['REQUEST_URI'];
        
        // Remove query string from URI
        $requestPath = parse_url($requestUri, PHP_URL_PATH);
        
        // Remove base path if present (handle different base paths)
        $scriptName = dirname($_SERVER['SCRIPT_NAME']);
        if ($scriptName !== '/' && strpos($requestPath, $scriptName) === 0) {
            $requestPath = substr($requestPath, strlen($scriptName));
        }
        
        // Also handle case where path might start with /api
        if (strpos($requestPath, '/api/') === 0) {
            $requestPath = substr($requestPath, 4); // Remove '/api'
        } elseif ($requestPath === '/api') {
            $requestPath = '';
        }
        
        // Remove leading/trailing slashes
        $requestPath = trim($requestPath, '/');
        
        // STEP 1: Check for exact string matches first (highest priority)
        // This ensures routes like /patients/last-registration-number match before /patients/{id}
        foreach ($this->routes as $route) {
            if ($route['method'] !== $requestMethod) {
                continue;
            }
            
            $routePath = trim($route['path'], '/');
            
            // Check for exact string match (no parameters, no regex)
            // Use strict comparison to ensure exact match
            if ($routePath === $requestPath && strpos($routePath, '{') === false) {
                // Exact match found - execute immediately
                $this->executeRoute($route, []);
                return;
            }
        }
        
        // STEP 2: Check parameterized routes only if no exact match was found
        foreach ($this->routes as $route) {
            if ($route['method'] !== $requestMethod) {
                continue;
            }
            
            $routePath = trim($route['path'], '/');
            
            // Skip if this route doesn't have parameters (already checked in STEP 1)
            if (strpos($routePath, '{') === false) {
                continue;
            }
            
            // Convert route path to regex pattern
            // Replace {param} with regex pattern that matches any non-slash characters
            $pattern = '#^' . preg_replace('#\{[^}]+\}#', '([^/]+)', $routePath) . '$#';
            
            // Check if path matches the pattern
            if (preg_match($pattern, $requestPath, $matches)) {
                // Remove first match (full string)
                array_shift($matches);
                
                $this->executeRoute($route, $matches);
                return;
            }
        }
        
        // No route matched
        Response::notFound('Endpoint not found');
    }
    
    /**
     * Execute a matched route
     * @param array $route Route configuration
     * @param array $params Route parameters
     */
    private function executeRoute($route, $params) {
        // Require authentication if needed
        if ($route['auth']) {
            AuthMiddleware::verify();
        }
        
        // Load controller
        $controllerName = $route['controller'];
        $actionName = $route['action'];
        
        if (!class_exists($controllerName)) {
            Response::serverError("Controller {$controllerName} not found");
        }
        
        // Instantiate controller and call action
        $controller = new $controllerName();
        
        if (!method_exists($controller, $actionName)) {
            Response::serverError("Method {$actionName} not found in {$controllerName}");
        }
        
        // Call controller action with route parameters
        call_user_func_array([$controller, $actionName], $params);
    }
}

