<?php
/**
 * Staff Model
 * Handles all database operations for staff table
 */

class StaffModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Get all staff members with pagination
     * @param int $limit Limit for pagination
     * @param int $offset Offset for pagination
     * @return array List of staff members
     */
    public function getAll($limit = null, $offset = 0) {
        if ($limit) {
            $stmt = $this->db->prepare("SELECT * FROM staff ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->bindValue(1, $limit, PDO::PARAM_INT);
            $stmt->bindValue(2, $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $stmt = $this->db->query("SELECT * FROM staff ORDER BY created_at DESC");
        }
        return $stmt->fetchAll();
    }
    
    /**
     * Get total count of staff members
     * @return int Total number of staff members
     */
    public function getTotalCount() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM staff");
        $result = $stmt->fetch();
        return (int) $result['total'];
    }
    
    /**
     * Get staff by ID
     * @param int $id Staff ID
     * @return array|false Staff data or false if not found
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM staff WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Create new staff member
     * @param array $data Staff data
     * @return int New staff ID
     */
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO staff (name, designation, department, mobile_number, email, address)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['name'],
            $data['designation'],
            $data['department'],
            $data['mobile_number'],
            $data['email'],
            $data['address']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update staff member
     * @param int $id Staff ID
     * @param array $data Updated staff data
     * @return bool True on success
     */
    public function update($id, $data) {
        $allowedFields = [
            'name', 'designation', 'department', 'mobile_number', 'email', 'address'
        ];
        
        $fields = [];
        $values = [];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false;
        }
        
        $values[] = $id;
        $sql = "UPDATE staff SET " . implode(', ', $fields) . " WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($values);
    }
    
    /**
     * Delete staff member
     * @param int $id Staff ID
     * @return bool True on success
     */
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM staff WHERE id = ?");
        return $stmt->execute([$id]);
    }
}

