<?php
/**
 * Patient Model
 * Handles all database operations for patients table
 */

class PatientModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Get all patients with pagination and category information
     * @param int $limit Limit for pagination
     * @param int $offset Offset for pagination
     * @return array List of patients with main category and sub category
     */
    public function getAll($limit = null, $offset = 0) {
        // Query to get patients with main category and sub category
        // Assumes patients.category stores category_id
        $sql = "
            SELECT 
                p.*,
                sub_cat.category_id as sub_category_id,
                sub_cat.name as sub_category_name,
                sub_cat.description as sub_category_description,
                main_cat.category_id as main_category_id,
                main_cat.name as main_category_name,
                main_cat.description as main_category_description
            FROM patients p
            LEFT JOIN category sub_cat ON p.category = sub_cat.category_id
            LEFT JOIN category main_cat ON sub_cat.parent_id = main_cat.category_id AND sub_cat.parent_id > 0
            ORDER BY p.id ASC
        ";
        $stmt = $this->db->query($sql);
       
        
        $patients = $stmt->fetchAll();
        
        // Format the results to include category information in a structured way
        return array_map(function($patient) {
            // Add main_category object (null if no parent category)
            $patient['main_category'] = null;
            if (!empty($patient['main_category_id'])) {
                $patient['main_category'] = [
                    'category_id' => $patient['main_category_id'],
                    'name' => $patient['main_category_name'],
                    'description' => $patient['main_category_description']
                ];
            }
            
            // Add sub_category object (null if no category assigned)
            $patient['sub_category'] = null;
            if (!empty($patient['sub_category_id'])) {
                $patient['sub_category'] = [
                    'category_id' => $patient['sub_category_id'],
                    'name' => $patient['sub_category_name'],
                    'description' => $patient['sub_category_description']
                ];
            }
            
            // Remove individual category fields to keep response clean
            unset($patient['sub_category_id']);
            unset($patient['sub_category_name']);
            unset($patient['sub_category_description']);
            unset($patient['main_category_id']);
            unset($patient['main_category_name']);
            unset($patient['main_category_description']);
            
            return $patient;
        }, $patients);
    }
    
    /**
     * Get total count of patients
     * @return int Total number of patients
     */
    public function getTotalCount() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM patients");
        $result = $stmt->fetch();
        return (int) $result['total'];
    }
    
    /**
     * Get patient by ID with category information
     * @param int $id Patient ID
     * @return array|false Patient data with main category and sub category, or false if not found
     */
    public function getById($id) {
        $sql = "
            SELECT 
                p.*,
                sub_cat.category_id as sub_category_id,
                sub_cat.name as sub_category_name,
                sub_cat.description as sub_category_description,
                main_cat.category_id as main_category_id,
                main_cat.name as main_category_name,
                main_cat.description as main_category_description
            FROM patients p
            LEFT JOIN category sub_cat ON p.category = sub_cat.category_id
            LEFT JOIN category main_cat ON sub_cat.parent_id = main_cat.category_id AND sub_cat.parent_id > 0
            WHERE p.id = ?
        ";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$id]);
        $patient = $stmt->fetch();
        
        if (!$patient) {
            return false;
        }
        
        // Format the result to include category information in a structured way
        // Add main_category object (null if no parent category)
        $patient['main_category'] = null;
        if (!empty($patient['main_category_id'])) {
            $patient['main_category'] = [
                'category_id' => $patient['main_category_id'],
                'name' => $patient['main_category_name'],
                'description' => $patient['main_category_description']
            ];
        }
        
        // Add sub_category object (null if no category assigned)
        $patient['sub_category'] = null;
        if (!empty($patient['sub_category_id'])) {
            $patient['sub_category'] = [
                'category_id' => $patient['sub_category_id'],
                'name' => $patient['sub_category_name'],
                'description' => $patient['sub_category_description']
            ];
        }
        
        // Remove individual category fields to keep response clean
        unset($patient['sub_category_id']);
        unset($patient['sub_category_name']);
        unset($patient['sub_category_description']);
        unset($patient['main_category_id']);
        unset($patient['main_category_name']);
        unset($patient['main_category_description']);
        
        return $patient;
    }
    
    /**
     * Get patient by registration number
     * @param string $registrationNumber Registration number
     * @return array|false Patient data or false if not found
     */
    public function getByRegistrationNumber($registrationNumber) {
        $stmt = $this->db->prepare("SELECT * FROM patients WHERE registration_number = ?");
        $stmt->execute([$registrationNumber]);
        return $stmt->fetch();
    }
    
    /**
     * Create new patient
     * @param array $data Patient data
     * @return int New patient ID
     */
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO patients (
                registration_number, name, age, sex, home_name, place, district,
                pincode, state, designation, office_name, office_place,
                mobile_number, adhar, email_id, old_reg_number, category
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['registration_number'],
            $data['name'],
            $data['age'],
            $data['sex'],
            $data['home_name'] ?? null,
            $data['place'] ?? null,
            $data['district'] ?? null,
            $data['pincode'] ?? null,
            $data['state'] ?? null,
            $data['designation'] ?? null,
            $data['office_name'] ?? null,
            $data['office_place'] ?? null,
            $data['mobile_number'],
            $data['adhar'] ?? null,
            $data['email_id'] ?? null,
            $data['old_reg_number'] ?? null,
            $data['category'] ?? null
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update patient
     * @param int $id Patient ID
     * @param array $data Updated patient data
     * @return bool True on success
     */
    public function update($id, $data) {
        $allowedFields = [
            'name', 'age', 'sex', 'home_name', 'place', 'district', 'pincode',
            'state', 'designation', 'office_name', 'office_place', 'mobile_number',
            'adhar', 'email_id', 'old_reg_number', 'category'
        ];
        
        $fields = [];
        $values = [];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false;
        }
        
        $values[] = $id;
        $sql = "UPDATE patients SET " . implode(', ', $fields) . " WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($values);
    }
    
    /**
     * Delete patient
     * @param int $id Patient ID
     * @return bool True on success
     */
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM patients WHERE id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Search patients across multiple fields with category information
     * @param string $query Search query
     * @param int $limit Limit for pagination
     * @param int $offset Offset for pagination
     * @return array List of matching patients with main category and sub category
     */
    public function search($query, $limit = null, $offset = 0) {
        $searchTerm = "%{$query}%";
        
        $sql = "
            SELECT 
                p.*,
                sub_cat.category_id as sub_category_id,
                sub_cat.name as sub_category_name,
                sub_cat.description as sub_category_description,
                main_cat.category_id as main_category_id,
                main_cat.name as main_category_name,
                main_cat.description as main_category_description
            FROM patients p
            LEFT JOIN category sub_cat ON p.category = sub_cat.category_id
            LEFT JOIN category main_cat ON sub_cat.parent_id = main_cat.category_id AND sub_cat.parent_id > 0
            WHERE p.name LIKE ? 
               OR p.mobile_number LIKE ?
               OR p.email_id LIKE ?
               OR p.registration_number LIKE ?
               OR p.old_reg_number LIKE ?
               OR p.place LIKE ?
               OR p.district LIKE ?
               OR p.state LIKE ?
               OR p.home_name LIKE ?
               OR p.office_place LIKE ?
            ORDER BY p.name ASC
        ";
        
        if ($limit) {
            $sql .= " LIMIT ? OFFSET ?";
            $stmt = $this->db->prepare($sql);
            $stmt->bindValue(1, $searchTerm);
            $stmt->bindValue(2, $searchTerm);
            $stmt->bindValue(3, $searchTerm);
            $stmt->bindValue(4, $searchTerm);
            $stmt->bindValue(5, $searchTerm);
            $stmt->bindValue(6, $searchTerm);
            $stmt->bindValue(7, $searchTerm);
            $stmt->bindValue(8, $searchTerm);
            $stmt->bindValue(9, $searchTerm);
            $stmt->bindValue(10, $limit, PDO::PARAM_INT);
            $stmt->bindValue(11, $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                $searchTerm, $searchTerm, $searchTerm, $searchTerm,
                $searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm
            ]);
        }
        
        $patients = $stmt->fetchAll();
        
        // Format the results to include category information in a structured way
        return array_map(function($patient) {
            // Add main_category object (null if no parent category)
            $patient['main_category'] = null;
            if (!empty($patient['main_category_id'])) {
                $patient['main_category'] = [
                    'category_id' => $patient['main_category_id'],
                    'name' => $patient['main_category_name'],
                    'description' => $patient['main_category_description']
                ];
            }
            
            // Add sub_category object (null if no category assigned)
            $patient['sub_category'] = null;
            if (!empty($patient['sub_category_id'])) {
                $patient['sub_category'] = [
                    'category_id' => $patient['sub_category_id'],
                    'name' => $patient['sub_category_name'],
                    'description' => $patient['sub_category_description']
                ];
            }
            
            // Remove individual category fields to keep response clean
            unset($patient['sub_category_id']);
            unset($patient['sub_category_name']);
            unset($patient['sub_category_description']);
            unset($patient['main_category_id']);
            unset($patient['main_category_name']);
            unset($patient['main_category_description']);
            
            return $patient;
        }, $patients);
    }
    
    /**
     * Get total count of search results
     * @param string $query Search query
     * @return int Total number of matching patients
     */
    public function getSearchTotalCount($query) {
        $searchTerm = "%{$query}%";
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as total FROM patients 
            WHERE name LIKE ? 
               OR mobile_number LIKE ?
               OR email_id LIKE ?
               OR registration_number LIKE ?
               OR old_reg_number LIKE ?
               OR place LIKE ?
               OR district LIKE ?
               OR state LIKE ?
               OR home_name LIKE ?
               OR office_place LIKE ?
        ");
        
        $stmt->execute([
            $searchTerm, $searchTerm, $searchTerm, $searchTerm,
            $searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm
        ]);
        
        $result = $stmt->fetch();
        return (int) $result['total'];
    }
    /**
     * Get the last patient record.
     * Used to retrieve the latest registration_number for pre-population.
     * 
     * @return array|false Returns the last patient row or false if not found.
     */
    public function getLastRecord() {
        $stmt = $this->db->query("
            SELECT * FROM patients 
            ORDER BY id DESC 
            LIMIT 1
        ");
        return $stmt->fetch();
    }

    /**
     * Get the last registration number.
     * Used to pre-populate the next value for registration_number in UI.
     *
     * @return string|null Returns the last registration number or null if not available.
     */
    public function getLastRegistrationNumber() {
        $lastRecord = $this->getLastRecord();
        if ($lastRecord && isset($lastRecord['registration_number'])) {
            return $lastRecord['registration_number'];
        }
        return null;
    }
}

