<?php
/**
 * Doctor Model
 * Handles all database operations for doctors table
 */

class DoctorModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Get all doctors with pagination
     * @param int $limit Limit for pagination
     * @param int $offset Offset for pagination
     * @return array List of doctors
     */
    public function getAll($limit = null, $offset = 0) {
        if ($limit) {
            $stmt = $this->db->prepare("SELECT * FROM doctors ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->bindValue(1, $limit, PDO::PARAM_INT);
            $stmt->bindValue(2, $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $stmt = $this->db->query("SELECT * FROM doctors ORDER BY created_at DESC");
        }
        return $stmt->fetchAll();
    }
    
    /**
     * Get total count of doctors
     * @return int Total number of doctors
     */
    public function getTotalCount() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM doctors");
        $result = $stmt->fetch();
        return (int) $result['total'];
    }
    
    /**
     * Get doctor by ID
     * @param int $id Doctor ID
     * @return array|false Doctor data or false if not found
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM doctors WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Get doctor by registration number
     * @param string $registrationNumber Registration number
     * @return array|false Doctor data or false if not found
     */
    public function getByRegistrationNumber($registrationNumber) {
        $stmt = $this->db->prepare("SELECT * FROM doctors WHERE registration_number = ?");
        $stmt->execute([$registrationNumber]);
        return $stmt->fetch();
    }
    
    /**
     * Create new doctor
     * @param array $data Doctor data
     * @return int New doctor ID
     */
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO doctors (name, specialization, qualification, registration_number, mobile_number, email, address)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['name'],
            $data['specialization'],
            $data['qualification'],
            $data['registration_number'],
            $data['mobile_number'],
            $data['email'],
            $data['address']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update doctor
     * @param int $id Doctor ID
     * @param array $data Updated doctor data
     * @return bool True on success
     */
    public function update($id, $data) {
        $allowedFields = [
            'name', 'specialization', 'qualification', 'registration_number',
            'mobile_number', 'email', 'address'
        ];
        
        $fields = [];
        $values = [];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false;
        }
        
        $values[] = $id;
        $sql = "UPDATE doctors SET " . implode(', ', $fields) . " WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($values);
    }
    
    /**
     * Delete doctor
     * @param int $id Doctor ID
     * @return bool True on success
     */
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM doctors WHERE id = ?");
        return $stmt->execute([$id]);
    }
}

