<?php
/**
 * Category Model
 * Handles all database operations for category table
 */

class CategoryModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Get all categories with pagination
     * @param int $limit Limit for pagination
     * @param int $offset Offset for pagination
     * @return array List of categories
     */
    public function getAll($limit = null, $offset = 0) {
        if ($limit) {
            $stmt = $this->db->prepare("SELECT * FROM category ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->bindValue(1, $limit, PDO::PARAM_INT);
            $stmt->bindValue(2, $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $stmt = $this->db->query("SELECT * FROM category ORDER BY created_at DESC");
        }
        return $stmt->fetchAll();
    }
    
    /**
     * Get total count of categories
     * @return int Total number of categories
     */
    public function getTotalCount() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM category");
        $result = $stmt->fetch();
        return (int) $result['total'];
    }
    
    /**
     * Get category by ID
     * @param int $id Category ID
     * @return array|false Category data or false if not found
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM category WHERE category_id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Get categories by parent ID
     * @param int $parentId Parent category ID
     * @return array List of child categories
     */
    public function getByParentId($parentId) {
        $stmt = $this->db->prepare("SELECT * FROM category WHERE parent_id = ? ORDER BY name ASC");
        $stmt->execute([$parentId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Create new category
     * @param array $data Category data
     * @return int New category ID
     */
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO category (name, parent_id, description)
            VALUES (?, ?, ?)
        ");
        
        $stmt->execute([
            $data['name'],
            $data['parent_id'] ?? 0,
            $data['description'] ?? null
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update category
     * @param int $id Category ID
     * @param array $data Updated category data
     * @return bool True on success
     */
    public function update($id, $data) {
        $allowedFields = ['name', 'parent_id', 'description'];
        
        $fields = [];
        $values = [];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false;
        }
        
        $values[] = $id;
        $sql = "UPDATE category SET " . implode(', ', $fields) . " WHERE category_id = ?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($values);
    }
    
    /**
     * Delete category
     * @param int $id Category ID
     * @return bool True on success
     */
    public function delete($id) {
        // Check if category has children
        $children = $this->getByParentId($id);
        if (!empty($children)) {
            return false; // Cannot delete category with children
        }
        
        $stmt = $this->db->prepare("DELETE FROM category WHERE category_id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Check if category exists
     * @param int $id Category ID
     * @return bool True if exists
     */
    public function exists($id) {
        $category = $this->getById($id);
        return $category !== false;
    }
}

