<?php
/**
 * Case Sheet Model
 * Handles all database operations for case_sheets table
 */

class CaseSheetModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Get case sheet by ID
     * @param int $id Case sheet ID
     * @return array|false Case sheet data or false if not found
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM case_sheets WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Get all case sheets for a patient
     * @param int $patientId Patient ID
     * @return array List of case sheets
     */
    public function getByPatientId($patientId) {
        $stmt = $this->db->prepare("SELECT * FROM case_sheets WHERE patient_id = ? ORDER BY date DESC");
        $stmt->execute([$patientId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Create new case sheet
     * @param array $data Case sheet data
     * @return int New case sheet ID
     */
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO case_sheets (patient_id, patient_name, date, chief_complaint, history, examination, diagnosis, treatment, doctor_name)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['patient_id'],
            $data['patient_name'],
            $data['date'],
            $data['chief_complaint'],
            $data['history'] ?? null,
            $data['examination'] ?? null,
            $data['diagnosis'] ?? null,
            $data['treatment'] ?? null,
            $data['doctor_name']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update case sheet
     * @param int $id Case sheet ID
     * @param array $data Updated case sheet data
     * @return bool True on success
     */
    public function update($id, $data) {
        $allowedFields = [
            'patient_id', 'patient_name', 'date', 'chief_complaint',
            'history', 'examination', 'diagnosis', 'treatment', 'doctor_name'
        ];
        
        $fields = [];
        $values = [];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false;
        }
        
        $values[] = $id;
        $sql = "UPDATE case_sheets SET " . implode(', ', $fields) . " WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($values);
    }
    
    /**
     * Delete case sheet
     * @param int $id Case sheet ID
     * @return bool True on success
     */
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM case_sheets WHERE id = ?");
        return $stmt->execute([$id]);
    }
}

