<?php
/**
 * Staff Controller
 * Handles all staff-related API endpoints
 */

class StaffController {
    private $staffModel;
    
    public function __construct() {
        $this->staffModel = new StaffModel();
    }
    
    /**
     * Get all staff members with pagination
     * GET /staff?page=1&limit=20
     */
    public function index() {
        try {
            // Get pagination parameters from query string
            $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
            $limit = isset($_GET['limit']) ? (int) $_GET['limit'] : DEFAULT_PAGE_SIZE;
            
            // Validate pagination parameters
            if ($page < 1) {
                $page = 1;
            }
            if ($limit < 1) {
                $limit = DEFAULT_PAGE_SIZE;
            }
            if ($limit > MAX_PAGE_SIZE) {
                $limit = MAX_PAGE_SIZE;
            }
            
            // Calculate offset
            $offset = ($page - 1) * $limit;
            
            // Get total count and paginated data
            $total = $this->staffModel->getTotalCount();
            $staff = $this->staffModel->getAll($limit, $offset);
            
            // Return response with data and pagination info
            Response::success('Staff retrieved successfully', [
                'data' => $staff,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'limit' => $limit,
                    'total_pages' => ceil($total / $limit)
                ]
            ]);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch staff: ' . $e->getMessage());
        }
    }
    
    /**
     * Get staff by ID
     * GET /staff/{id}
     */
    public function show($id) {
        try {
            $staffMember = $this->staffModel->getById($id);
            
            if (!$staffMember) {
                Response::notFound('Staff not found');
            }
            
            Response::success('Staff retrieved successfully', $staffMember);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch staff: ' . $e->getMessage());
        }
    }
    
    /**
     * Create new staff member
     * POST /staff
     */
    public function create() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $requiredFields = ['name', 'designation', 'department', 'mobile_number', 'email', 'address'];
        $errors = Validator::validateRequired($data, $requiredFields);
        
        // Validate email
        if (!empty($data['email']) && !Validator::validateEmail($data['email'])) {
            $errors['email'] = ['Invalid email format'];
        }
        
        // Validate mobile number
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            $errors['mobile_number'] = ['Mobile number must be 10 digits'];
        }
        
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        try {
            $staffId = $this->staffModel->create($data);
            $staffMember = $this->staffModel->getById($staffId);
            Response::success('Staff created successfully', $staffMember, 201);
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                Response::error('Email already exists', 400);
            }
            Response::serverError('Failed to create staff: ' . $e->getMessage());
        }
    }
    
    /**
     * Update staff member
     * PUT /staff/{id}
     */
    public function update($id) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Check if staff exists
        $staffMember = $this->staffModel->getById($id);
        if (!$staffMember) {
            Response::notFound('Staff not found');
        }
        
        // Validate email if provided
        if (!empty($data['email']) && !Validator::validateEmail($data['email'])) {
            Response::validationError(['email' => ['Invalid email format']]);
        }
        
        // Validate mobile number if provided
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            Response::validationError(['mobile_number' => ['Mobile number must be 10 digits']]);
        }
        
        try {
            $updated = $this->staffModel->update($id, $data);
            
            if (!$updated) {
                Response::error('No fields to update', 400);
            }
            
            $staffMember = $this->staffModel->getById($id);
            Response::success('Staff updated successfully', $staffMember);
        } catch (Exception $e) {
            Response::serverError('Failed to update staff: ' . $e->getMessage());
        }
    }
    
    /**
     * Delete staff member
     * DELETE /staff/{id}
     */
    public function delete($id) {
        try {
            $staffMember = $this->staffModel->getById($id);
            
            if (!$staffMember) {
                Response::notFound('Staff not found');
            }
            
            $this->staffModel->delete($id);
            Response::success('Staff deleted successfully');
        } catch (Exception $e) {
            Response::serverError('Failed to delete staff: ' . $e->getMessage());
        }
    }
}

