<?php
/**
 * Patient Controller
 * Handles all patient-related API endpoints
 */

class PatientController {
    private $patientModel;
    
    public function __construct() {
        $this->patientModel = new PatientModel();
    }
    
    /**
     * Get all patients with pagination
     * GET /patients?page=1&limit=20
     */
    public function index() {
        try {
            // Get pagination parameters from query string
            $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
            $limit = isset($_GET['limit']) ? (int) $_GET['limit'] : DEFAULT_PAGE_SIZE;
            
            // Validate pagination parameters
            if ($page < 1) {
                $page = 1;
            }
            if ($limit < 1) {
                $limit = DEFAULT_PAGE_SIZE;
            }
            if ($limit > MAX_PAGE_SIZE) {
                $limit = MAX_PAGE_SIZE;
            }
            
            // Calculate offset
            $offset = ($page - 1) * $limit;
            
            // Get total count and paginated data
            $total = $this->patientModel->getTotalCount();
            $patients = $this->patientModel->getAll($limit, $offset);
            
            // Return response with data and pagination info
            Response::success('Patients retrieved successfully', [
                'data' => $patients,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'limit' => $limit,
                    'total_pages' => ceil($total / $limit)
                ]
            ]);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch patients: ' . $e->getMessage());
        }
    }
    
    /**
     * Get patient by ID
     * GET /patients/{id}
     */
    public function show($id) {
        try {
            $patient = $this->patientModel->getById($id);
            
            if (!$patient) {
                Response::notFound('Patient not found');
            }
            
            Response::success('Patient retrieved successfully', $patient);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch patient: ' . $e->getMessage());
        }
    }
    
    /**
     * Create new patient
     * POST /patients
     */
    public function create() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $requiredFields = ['registration_number', 'name', 'age', 'sex', 'mobile_number'];
        $errors = Validator::validateRequired($data, $requiredFields);
        
        // Validate email if provided
        if (!empty($data['email_id']) && !Validator::validateEmail($data['email_id'])) {
            $errors['email_id'] = ['Invalid email format'];
        }
        
        // Validate mobile number
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            $errors['mobile_number'] = ['Mobile number must be 10 digits'];
        }
        
        // Validate age
        if (!empty($data['age']) && !Validator::validateAge($data['age'])) {
            $errors['age'] = ['Age must be between 1 and 150'];
        }
        
        // Validate sex enum
        if (!empty($data['sex']) && !Validator::validateEnum($data['sex'], ['Male', 'Female', 'Other'])) {
            $errors['sex'] = ['Sex must be Male, Female, or Other'];
        }
        
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        // Check if registration number already exists
        $existing = $this->patientModel->getByRegistrationNumber($data['registration_number']);
        if ($existing) {
            Response::error('Registration number already exists', 400);
        }
        
        try {
            $patientId = $this->patientModel->create($data);
            $patient = $this->patientModel->getById($patientId);
            Response::success('Patient created successfully', $patient, 201);
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                Response::error('Registration number already exists', 400);
            }
            Response::serverError('Failed to create patient: ' . $e->getMessage());
        }
    }
    
    /**
     * Update patient
     * PUT /patients/{id}
     */
    public function update($id) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Check if patient exists
        $patient = $this->patientModel->getById($id);
        if (!$patient) {
            Response::notFound('Patient not found');
        }
        
        // Validate email if provided
        if (!empty($data['email_id']) && !Validator::validateEmail($data['email_id'])) {
            Response::validationError(['email_id' => ['Invalid email format']]);
        }
        
        // Validate mobile number if provided
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            Response::validationError(['mobile_number' => ['Mobile number must be 10 digits']]);
        }
        
        // Validate age if provided
        if (!empty($data['age']) && !Validator::validateAge($data['age'])) {
            Response::validationError(['age' => ['Age must be between 1 and 150']]);
        }
        
        // Validate sex enum if provided
        if (!empty($data['sex']) && !Validator::validateEnum($data['sex'], ['Male', 'Female', 'Other'])) {
            Response::validationError(['sex' => ['Sex must be Male, Female, or Other']]);
        }
        
        try {
            $updated = $this->patientModel->update($id, $data);
            
            if (!$updated) {
                Response::error('No fields to update', 400);
            }
            
            $patient = $this->patientModel->getById($id);
            Response::success('Patient updated successfully', $patient);
        } catch (Exception $e) {
            Response::serverError('Failed to update patient: ' . $e->getMessage());
        }
    }
    
    /**
     * Delete patient
     * DELETE /patients/{id}
     */
    public function delete($id) {
        try {
            $patient = $this->patientModel->getById($id);
            
            if (!$patient) {
                Response::notFound('Patient not found');
            }
            
            $this->patientModel->delete($id);
            Response::success('Patient deleted successfully');
        } catch (Exception $e) {
            Response::serverError('Failed to delete patient: ' . $e->getMessage());
        }
    }
    
    /**
     * Search patients across multiple fields
     * GET /patients/search?q={query}&page=1&limit=20
     * 
     * Searches in: name, address (place, district, state, home_name, office_place), 
     * registration_number, old_reg_number, email, phone (mobile_number)
     */
    public function search() {
        $query = $_GET['q'] ?? '';
        
        if (empty($query)) {
            Response::error('Search query is required', 400);
        }
        
        try {
            // Get pagination parameters from query string
            $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
            $limit = isset($_GET['limit']) ? (int) $_GET['limit'] : DEFAULT_PAGE_SIZE;
            
            // Validate pagination parameters
            if ($page < 1) {
                $page = 1;
            }
            if ($limit < 1) {
                $limit = DEFAULT_PAGE_SIZE;
            }
            if ($limit > MAX_PAGE_SIZE) {
                $limit = MAX_PAGE_SIZE;
            }
            
            // Calculate offset
            $offset = ($page - 1) * $limit;
            
            // Get total count and paginated search results
            $total = $this->patientModel->getSearchTotalCount($query);
            $patients = $this->patientModel->search($query, $limit, $offset);
            
            // Return response with data and pagination info (same format as index)
            Response::success('Search completed', [
                'data' => $patients,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'limit' => $limit,
                    'total_pages' => ceil($total / $limit)
                ]
            ]);
        } catch (Exception $e) {
            Response::serverError('Search failed: ' . $e->getMessage());
        }
    }
    /**
     * Get the last registration number for pre-population
     * GET /patients/last-registration-number
     */
    public function getLastRegistrationNumber() {
        try {
            $lastRegistrationNumber = $this->patientModel->getLastRegistrationNumber();
            Response::success('Last registration number retrieved successfully', $lastRegistrationNumber);
        } catch (Exception $e) {
            Response::serverError('Failed to get last registration number: ' . $e->getMessage());
        }
    }
}

