<?php
/**
 * Doctor Controller
 * Handles all doctor-related API endpoints
 */

class DoctorController {
    private $doctorModel;
    
    public function __construct() {
        $this->doctorModel = new DoctorModel();
    }
    
    /**
     * Get all doctors with pagination
     * GET /doctors?page=1&limit=20
     */
    public function index() {
        try {
            // Get pagination parameters from query string
            $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
            $limit = isset($_GET['limit']) ? (int) $_GET['limit'] : DEFAULT_PAGE_SIZE;
            
            // Validate pagination parameters
            if ($page < 1) {
                $page = 1;
            }
            if ($limit < 1) {
                $limit = DEFAULT_PAGE_SIZE;
            }
            if ($limit > MAX_PAGE_SIZE) {
                $limit = MAX_PAGE_SIZE;
            }
            
            // Calculate offset
            $offset = ($page - 1) * $limit;
            
            // Get total count and paginated data
            $total = $this->doctorModel->getTotalCount();
            $doctors = $this->doctorModel->getAll($limit, $offset);
            
            // Return response with data and pagination info
            Response::success('Doctors retrieved successfully', [
                'data' => $doctors,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'limit' => $limit,
                    'total_pages' => ceil($total / $limit)
                ]
            ]);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch doctors: ' . $e->getMessage());
        }
    }
    
    /**
     * Get doctor by ID
     * GET /doctors/{id}
     */
    public function show($id) {
        try {
            $doctor = $this->doctorModel->getById($id);
            
            if (!$doctor) {
                Response::notFound('Doctor not found');
            }
            
            Response::success('Doctor retrieved successfully', $doctor);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch doctor: ' . $e->getMessage());
        }
    }
    
    /**
     * Create new doctor
     * POST /doctors
     */
    public function create() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $requiredFields = ['name', 'specialization', 'qualification', 'registration_number', 'mobile_number', 'email', 'address'];
        $errors = Validator::validateRequired($data, $requiredFields);
        
        // Validate email
        if (!empty($data['email']) && !Validator::validateEmail($data['email'])) {
            $errors['email'] = ['Invalid email format'];
        }
        
        // Validate mobile number
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            $errors['mobile_number'] = ['Mobile number must be 10 digits'];
        }
        
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        // Check if registration number already exists
        $existing = $this->doctorModel->getByRegistrationNumber($data['registration_number']);
        if ($existing) {
            Response::error('Registration number already exists', 400);
        }
        
        try {
            $doctorId = $this->doctorModel->create($data);
            $doctor = $this->doctorModel->getById($doctorId);
            Response::success('Doctor created successfully', $doctor, 201);
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                Response::error('Registration number or email already exists', 400);
            }
            Response::serverError('Failed to create doctor: ' . $e->getMessage());
        }
    }
    
    /**
     * Update doctor
     * PUT /doctors/{id}
     */
    public function update($id) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Check if doctor exists
        $doctor = $this->doctorModel->getById($id);
        if (!$doctor) {
            Response::notFound('Doctor not found');
        }
        
        // Validate email if provided
        if (!empty($data['email']) && !Validator::validateEmail($data['email'])) {
            Response::validationError(['email' => ['Invalid email format']]);
        }
        
        // Validate mobile number if provided
        if (!empty($data['mobile_number']) && !Validator::validateMobile($data['mobile_number'])) {
            Response::validationError(['mobile_number' => ['Mobile number must be 10 digits']]);
        }
        
        try {
            $updated = $this->doctorModel->update($id, $data);
            
            if (!$updated) {
                Response::error('No fields to update', 400);
            }
            
            $doctor = $this->doctorModel->getById($id);
            Response::success('Doctor updated successfully', $doctor);
        } catch (Exception $e) {
            Response::serverError('Failed to update doctor: ' . $e->getMessage());
        }
    }
    
    /**
     * Delete doctor
     * DELETE /doctors/{id}
     */
    public function delete($id) {
        try {
            $doctor = $this->doctorModel->getById($id);
            
            if (!$doctor) {
                Response::notFound('Doctor not found');
            }
            
            $this->doctorModel->delete($id);
            Response::success('Doctor deleted successfully');
        } catch (Exception $e) {
            Response::serverError('Failed to delete doctor: ' . $e->getMessage());
        }
    }
}

