<?php
/**
 * Category Controller
 * Handles all category-related API endpoints
 */

class CategoryController {
    private $categoryModel;
    
    public function __construct() {
        $this->categoryModel = new CategoryModel();
    }
    
    /**
     * Get all categories with pagination
     * GET /categories?page=1&limit=20
     */
    public function index() {
        try {
            // Get pagination parameters from query string
            $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
            $limit = isset($_GET['limit']) ? (int) $_GET['limit'] : DEFAULT_PAGE_SIZE;
            
            // Validate pagination parameters
            if ($page < 1) {
                $page = 1;
            }
            if ($limit < 1) {
                $limit = DEFAULT_PAGE_SIZE;
            }
            if ($limit > MAX_PAGE_SIZE) {
                $limit = MAX_PAGE_SIZE;
            }
            
            // Calculate offset
            $offset = ($page - 1) * $limit;
            
            // Get total count and paginated data
            $total = $this->categoryModel->getTotalCount();
            $categories = $this->categoryModel->getAll($limit, $offset);
            
            // Return response with data and pagination info
            Response::success('Categories retrieved successfully', [
                'data' => $categories,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'limit' => $limit,
                    'total_pages' => ceil($total / $limit)
                ]
            ]);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch categories: ' . $e->getMessage());
        }
    }
    
    /**
     * Get category by ID
     * GET /categories/{id}
     */
    public function show($id) {
        try {
            $category = $this->categoryModel->getById($id);
            
            if (!$category) {
                Response::notFound('Category not found');
            }
            
            Response::success('Category retrieved successfully', $category);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch category: ' . $e->getMessage());
        }
    }
    
    /**
     * Get categories by parent ID
     * GET /categories/parent/{parent_id}
     */
    public function getByParent($parentId) {
        try {
            $categories = $this->categoryModel->getByParentId($parentId);
            Response::success('Categories retrieved successfully', $categories);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch categories: ' . $e->getMessage());
        }
    }
    
    /**
     * Create new category
     * POST /categories
     */
    public function create() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $requiredFields = ['name'];
        $errors = Validator::validateRequired($data, $requiredFields);
        
        // Validate parent_id if provided (must be valid category ID or 0)
        if (isset($data['parent_id'])) {
            $parentId = (int) $data['parent_id'];
            if ($parentId > 0 && !$this->categoryModel->exists($parentId)) {
                $errors['parent_id'] = ['Parent category does not exist'];
            }
            // Prevent setting parent to itself (will be checked after creation)
        }
        
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        try {
            // Ensure parent_id defaults to 0 if not provided
            if (!isset($data['parent_id'])) {
                $data['parent_id'] = 0;
            }
            
            $categoryId = $this->categoryModel->create($data);
            $category = $this->categoryModel->getById($categoryId);
            Response::success('Category created successfully', $category, 201);
        } catch (PDOException $e) {
            Response::serverError('Failed to create category: ' . $e->getMessage());
        }
    }
    
    /**
     * Update category
     * PUT /categories/{id}
     */
    public function update($id) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Check if category exists
        $category = $this->categoryModel->getById($id);
        if (!$category) {
            Response::notFound('Category not found');
        }
        
        // Validate parent_id if provided
        if (isset($data['parent_id'])) {
            $parentId = (int) $data['parent_id'];
            
            // Prevent setting parent to itself
            if ($parentId == $id) {
                Response::validationError(['parent_id' => ['Category cannot be its own parent']]);
            }
            
            // If parent_id is not 0, verify it exists
            if ($parentId > 0 && !$this->categoryModel->exists($parentId)) {
                Response::validationError(['parent_id' => ['Parent category does not exist']]);
            }
            
            // Prevent circular references (check if trying to set parent to a child)
            $children = $this->categoryModel->getByParentId($id);
            foreach ($children as $child) {
                if ($child['category_id'] == $parentId) {
                    Response::validationError(['parent_id' => ['Cannot set parent to a child category']]);
                }
            }
        }
        
        try {
            $updated = $this->categoryModel->update($id, $data);
            
            if (!$updated) {
                Response::error('No fields to update', 400);
            }
            
            $category = $this->categoryModel->getById($id);
            Response::success('Category updated successfully', $category);
        } catch (Exception $e) {
            Response::serverError('Failed to update category: ' . $e->getMessage());
        }
    }
    
    /**
     * Delete category
     * DELETE /categories/{id}
     */
    public function delete($id) {
        try {
            $category = $this->categoryModel->getById($id);
            
            if (!$category) {
                Response::notFound('Category not found');
            }
            
            // Check if category has children
            $children = $this->categoryModel->getByParentId($id);
            if (!empty($children)) {
                Response::error('Cannot delete category with child categories. Please delete or move child categories first.', 400);
            }
            
            $deleted = $this->categoryModel->delete($id);
            
            if ($deleted) {
                Response::success('Category deleted successfully');
            } else {
                Response::serverError('Failed to delete category');
            }
        } catch (Exception $e) {
            Response::serverError('Failed to delete category: ' . $e->getMessage());
        }
    }
}

