<?php
/**
 * Case Sheet Controller
 * Handles all case sheet-related API endpoints
 */

class CaseSheetController {
    private $caseSheetModel;
    private $patientModel;
    
    public function __construct() {
        $this->caseSheetModel = new CaseSheetModel();
        $this->patientModel = new PatientModel();
    }
    
    /**
     * Get case sheet by ID
     * GET /case-sheets/{id}
     */
    public function show($id) {
        try {
            $caseSheet = $this->caseSheetModel->getById($id);
            
            if (!$caseSheet) {
                Response::notFound('Case sheet not found');
            }
            
            Response::success('Case sheet retrieved successfully', $caseSheet);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch case sheet: ' . $e->getMessage());
        }
    }
    
    /**
     * Get all case sheets for a patient
     * GET /case-sheets/patient/{patient_id}
     */
    public function getByPatient($patientId) {
        try {
            // Verify patient exists
            $patient = $this->patientModel->getById($patientId);
            if (!$patient) {
                Response::notFound('Patient not found');
            }
            
            $caseSheets = $this->caseSheetModel->getByPatientId($patientId);
            Response::success('Case sheets retrieved successfully', $caseSheets);
        } catch (Exception $e) {
            Response::serverError('Failed to fetch case sheets: ' . $e->getMessage());
        }
    }
    
    /**
     * Create new case sheet
     * POST /case-sheets
     */
    public function create() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $requiredFields = ['patient_id', 'patient_name', 'date', 'chief_complaint', 'doctor_name'];
        $errors = Validator::validateRequired($data, $requiredFields);
        
        // Validate date format
        if (!empty($data['date']) && !Validator::validateDate($data['date'])) {
            $errors['date'] = ['Invalid date format. Use YYYY-MM-DD or YYYY-MM-DD HH:mm:ss'];
        }
        
        // Verify patient exists
        if (!empty($data['patient_id'])) {
            $patient = $this->patientModel->getById($data['patient_id']);
            if (!$patient) {
                $errors['patient_id'] = ['Patient not found'];
            }
        }
        
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        try {
            $caseSheetId = $this->caseSheetModel->create($data);
            $caseSheet = $this->caseSheetModel->getById($caseSheetId);
            Response::success('Case sheet created successfully', $caseSheet, 201);
        } catch (Exception $e) {
            Response::serverError('Failed to create case sheet: ' . $e->getMessage());
        }
    }
    
    /**
     * Update case sheet
     * PUT /case-sheets/{id}
     */
    public function update($id) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Check if case sheet exists
        $caseSheet = $this->caseSheetModel->getById($id);
        if (!$caseSheet) {
            Response::notFound('Case sheet not found');
        }
        
        // Validate date format if provided
        if (!empty($data['date']) && !Validator::validateDate($data['date'])) {
            Response::validationError(['date' => ['Invalid date format. Use YYYY-MM-DD or YYYY-MM-DD HH:mm:ss']]);
        }
        
        // Verify patient exists if patient_id is being updated
        if (!empty($data['patient_id'])) {
            $patient = $this->patientModel->getById($data['patient_id']);
            if (!$patient) {
                Response::validationError(['patient_id' => ['Patient not found']]);
            }
        }
        
        try {
            $updated = $this->caseSheetModel->update($id, $data);
            
            if (!$updated) {
                Response::error('No fields to update', 400);
            }
            
            $caseSheet = $this->caseSheetModel->getById($id);
            Response::success('Case sheet updated successfully', $caseSheet);
        } catch (Exception $e) {
            Response::serverError('Failed to update case sheet: ' . $e->getMessage());
        }
    }
    
    /**
     * Delete case sheet
     * DELETE /case-sheets/{id}
     */
    public function delete($id) {
        try {
            $caseSheet = $this->caseSheetModel->getById($id);
            
            if (!$caseSheet) {
                Response::notFound('Case sheet not found');
            }
            
            $this->caseSheetModel->delete($id);
            Response::success('Case sheet deleted successfully');
        } catch (Exception $e) {
            Response::serverError('Failed to delete case sheet: ' . $e->getMessage());
        }
    }
}

