<?php
/**
 * Authentication Controller
 * Handles user authentication (login, logout)
 */

class AuthController {
    private $userModel;
    
    public function __construct() {
        $this->userModel = new UserModel();
    }
    
    /**
     * Handle user login
     * POST /auth/login
     */
    public function login() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $errors = Validator::validateRequired($data, ['username', 'password']);
        if (!empty($errors)) {
            Response::validationError($errors);
        }
        
        $username = $data['username'];
        $password = $data['password'];
        
        try {
            // Find user by username or email
            $user = $this->userModel->findByUsernameOrEmail($username);
           
            if (!$user || !password_verify($password, $user['password'])) {
                Response::error('Invalid credentials', 401);
            }
            
            // Generate secure token
            $token = bin2hex(random_bytes(32));
            
            // Store token in database
            $this->userModel->updateToken($user['id'], $token);
            
            // Return success response with user data and token
            Response::success('Login successful', [
                'user' => [
                    'id' => $user['id'],
                    'username' => $user['username'],
                    'email' => $user['email'],
                    'role' => $user['role']
                ],
                'token' => $token
            ]);
        } catch (Exception $e) {
            Response::serverError('Login failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Handle user logout
     * POST /auth/logout
     */
    public function logout() {
        // Get authenticated user from middleware
        $user = AuthMiddleware::verify();
        
        try {
            // Clear token from database
            $this->userModel->clearToken($user['id']);
            Response::success('Logout successful');
        } catch (Exception $e) {
            Response::serverError('Logout failed: ' . $e->getMessage());
        }
    }
}

